#include "pch.h"

#include "Common.hpp"
#include "FTNStruc.hpp"
#include "CfgParse.hpp"
#include "Misc.hpp"
#include "BSO.hpp"
#include "Adrenalin.h"
#include "crc32.hpp"
#include "Tic.hpp"
#include "ticlexems.hpp"
#include "LinksManager.h"

/*
 * Uncomment the definition below to use strict fsc accordance
 * algorithm
 * 
 */

//#define STRICT_FSC_MAPPING

// ---------
// CTicNamer
// ---------

CTicNamer::CTicNamer() {
    resetFields();
}

CTicNamer::CTicNamer(LPCTSTR szSeekPath) {
    resetFields();
    readTICs( szSeekPath );
}

CTicNamer::~CTicNamer() {
    freeFields();
}

void CTicNamer::resetFields() {
    intervals       = NULL;
    intervalsCount  = 0;
    freeNumber      = -1;
    currentInterval = -1;
}

void CTicNamer::freeFields() {
    if (intervals != NULL && intervalsCount > 0)
        free(intervals);
    resetFields();
}

int compareIntegers(const void *p1, const void *p2) {
    int *i1, *i2;

    i1 = (int*)p1;
    i2 = (int*)p2;
    return ((*i1 < *i2) ? -1 : ((*i1 == *i2) ? 0 : 1));
}


void CTicNamer::collectTics(int*& numbers, int& numberCount, LPCTSTR path) {
	assert(numberCount >= 0 && path != NULL);

	TCHAR	mask[MAX_PATH];

	lstrcpy( mask, path );
	if (mask[lstrlen(mask) - 1] != TEXT('\\')) {
		lstrcat( mask, TEXT("\\") );
	}
	lstrcat( mask, TEXT("*.tic") );

	// prepare bigger buffer
	int	bufferSize;
	if (numberCount <= 0) {
		bufferSize = 1024;
	} else {
		bufferSize = numberCount*2;
	}
	int*	newNumbers = (int*)realloc( numbers, sizeof(int)*bufferSize );
	if (newNumbers == NULL) {
		// TODO: improve error processing
		throw "out of memory";
	}
	numbers = newNumbers;

	WIN32_FIND_DATA	findData;
	HANDLE			fileHandle = ::FindFirstFile( mask, &findData );
	if (fileHandle != INVALID_HANDLE_VALUE) {
		do {
			int	newNum = parseNumber( findData.cFileName );
			if (newNum >= 0) {
				if (numberCount >= bufferSize) {
					do {
						bufferSize *= 2;
					} while (bufferSize <= numberCount);
					newNumbers = (int*)realloc( numbers, sizeof(int)*bufferSize);
					if (newNumbers == NULL) {
						throw "out of memory";	// TODO: do better error-processing
					}
					numbers = newNumbers;
				}				
				numbers[numberCount] = newNum;
				++numberCount;
			}
		} while (::FindNextFile( fileHandle, &findData ));
		::FindClose( fileHandle );
	}

	// cut additional memory that is not used

	if (numberCount == 0) {
		free(numbers);
		numbers = NULL;
	} else {
		newNumbers = (int*)realloc( numbers, sizeof(int)*numberCount );
		if (newNumbers == NULL) {
			throw "out of memory";
		}
		numbers = newNumbers;
	}
}


void CTicNamer::prepareIntervals(int* numbers, int numberCount) {
    SNumberInterval *newIntervals;
    int              i, j, len;

    freeFields();

	// prepare address string

	TCHAR	szAddr[5+1+5+1+5+1+5+1+MAX_DOMAIN_SIZE+1];

	wsprintf( szAddr, "%d:%d/%d.%d@%s",
		 g_vars.m_addrMain.Zone, g_vars.m_addrMain.Net,
		 g_vars.m_addrMain.Node, g_vars.m_addrMain.Point,
		 g_vars.m_addrMain.Domain );

	UINT	crc = getBufferCRC32( szAddr, lstrlen(szAddr) );


    if (numberCount > 0) {
        qsort( (void*)numbers, numberCount, sizeof(int), compareIntegers );
        i = 0;
        while (i < numberCount) {
            len = 1;
            int newNumber = numbers[i];
            j = i+1;
            while (j < numberCount && numbers[j] == newNumber+1) {
                newNumber = numbers[j];
                len++;
                j++;
            }
            newIntervals = (SNumberInterval*)realloc( intervals,
                                                      (intervalsCount+1)*
                                                      sizeof(SNumberInterval));
            if (newIntervals != NULL) {
                intervals = newIntervals;
                intervals[intervalsCount].number = numbers[i];
                intervals[intervalsCount].length = len;
                intervalsCount++;
            }
            i += len;
        }
        
        freeNumber	= (((int)time(NULL) ^ crc)) & 0xEFFFF;

        currentInterval = 0;
        while (currentInterval < intervalsCount && freeNumber >= intervals[currentInterval].number) {
        	if (freeNumber < (intervals[currentInterval].number + intervals[currentInterval].length)) {
        		freeNumber = intervals[currentInterval].number + intervals[currentInterval].length;
        	}
       		currentInterval++;
        }

        if (freeNumber > 999999) {
        	freeNumber = 0;
        	currentInterval = 0;
        	if (intervals[0].number == 0) {
        		freeNumber = intervals[0].length;
        		currentInterval++;
        	}
        }

    }
    else {
        freeNumber      = (((int)time(NULL) ^ crc)) & 0xEFFFF;
        currentInterval = 1;
    }
}

void CTicNamer::readTICs(LPCTSTR szSeekPath) {
    int             *numbers;
    int              numberCount;

    numbers     = NULL;
    numberCount = 0;
    
	collectTics( numbers, numberCount, szSeekPath );

	prepareIntervals( numbers, numberCount );

	if (numbers != NULL) {
		free( numbers );
	}
}


void CTicNamer::collectAllOutputTics() {
	int*	numbers     = NULL;
	int		numberCount = 0;

	collectTics( numbers, numberCount, g_vars.m_szNewTicsPath );

	CommonDataPtr	commonData = CCommonData::getInstance();

	if (commonData->getFileBoxesPath().size() > 0) {
		tstring	mask = commonData->getFileBoxesPath();
		mask += TEXT("*.*");

		WIN32_FIND_DATA	findData;
		HANDLE	fileHandle = ::FindFirstFile( mask.c_str(), &findData );
		if (fileHandle != INVALID_HANDLE_VALUE) {
			do {
				// check only directories
				if ((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0) {
					// exclude special directories from collection process
					if (lstrcmp( findData.cFileName, TEXT(".") ) != 0 &&
						lstrcmp( findData.cFileName, TEXT("..") ) != 0)
					{
						collectTics( numbers, numberCount, (commonData->getFileBoxesPath() + findData.cFileName).c_str() );
					}
				}
			} while (::FindNextFile( fileHandle, &findData ));
			::FindClose( fileHandle );
		}
	}

	prepareIntervals( numbers, numberCount );

	if (numbers != NULL) {
		free( numbers );
	}
}


int CTicNamer::parseNumber(LPCTSTR szTicName) {
    int number;
    int count;

    // ticName is in format <letter><letter><6 numbers>.tic
    // its length is 12

    if (lstrlen(szTicName) != 12)
        return (-1);

    // skip 2 first letters
    szTicName += 2;

    number = 0;
    count  = 0;
    while (count < 6 && (*szTicName >= _T('0') && *szTicName <= _T('9'))) {
        number *= 10;
        number += *szTicName - _T('0');
        szTicName++;
        count++;
    }
    
    if (count != 6)
        return (-1);
    
    if (lstrcmpi( szTicName, _T(".tic") ) != 0)
        return (-1);

    return (number);
}


int CTicNamer::getFreeNumber() {
    int oldFreeNumber = freeNumber;

    if (freeNumber == 999999) {
    	freeNumber = 0;
    	currentInterval = 0;
    }
    else {
    	freeNumber++;
    }

    if (currentInterval < intervalsCount) {
        if (freeNumber == intervals[currentInterval].number) {
            freeNumber += intervals[currentInterval].length;
            if (freeNumber > 999999) {
            	freeNumber = 0;
            	currentInterval = 0;
        	if (intervals[0].number == 0) {
        		freeNumber = intervals[0].length;
        		currentInterval++;
        	}
            }
            else {
            	currentInterval++;
            }
        }
    }

    return (oldFreeNumber);
}

void CTicNamer::printDebugInfo() {
    int i;
    
    COUT << _T("Intervals count: ") << intervalsCount << endl;
    for (i = 0; i < intervalsCount; i++) {
        COUT << _T("interval ") << i << _T(": from ") << intervals[i].number
            << _T(", with length ") << intervals[i].length << endl;
    }
    COUT << _T("First 10 free numbers:\n");
    for (i = 0; i < 10; i++)
        COUT << getFreeNumber() << endl;
}


// uncomment next definition to see TIC information during parsering
// #define TIC_DEBUG_INFO

CTic::CTic() : m_BSOManager() {
	Free();
	m_TicNamer.collectAllOutputTics();
}

CTic::~CTic() {
	Free();
}

void CTic::Free() {
    *m_szArea      = _T('\0');
    *m_szAreaDesc  = _T('\0');
    *m_szFile      = _T('\0');
    *m_szFullName  = _T('\0');
    *m_szLongName  = _T('\0');
    m_bCRC32Present = false;
    m_uiCRC32      = 0;
    *m_szMagic     = _T('\0');
    *m_szReplaces  = _T('\0');
    *m_szDesc      = _T('\0');
    m_listLDesc.clear();
    m_iSize     = -1;
    memset( &m_addrOrigin, 0, sizeof(SFTNAddress) );
    memset( &m_addrFrom,   0, sizeof(SFTNAddress) );
    memset( &m_addrTo,     0, sizeof(SFTNAddress) );
    *m_szCreatedBy = _T('\0');
    m_listVia.clear();
    m_listPath.clear();
    m_listSeenby.clear();
    m_listUnknown.clear();
    *m_szPW = _T('\0');
    *m_szFromPwd = _T('\0');

    m_vsDescLines.clear();
}

bool CTic::ParseAddr(LPCTSTR s, SFTNAddress &Addr) {
    TCHAR	szFullAddr[24+MAX_DOMAIN_SIZE];
    TCHAR	szTempStr[24+MAX_DOMAIN_SIZE];
    TCHAR	szDomain[MAX_DOMAIN_SIZE];
    bool	ColonPresent, SlashPresent, PointPresent, DogPresent, GridPresent;
    int		DogPos, GridPos;
    int		i;

    if (lstrlen(s) >= 24+MAX_DOMAIN_SIZE)
        return (false);
    
    GridPresent = ColonPresent = SlashPresent = PointPresent = DogPresent = false;
    i = 0;
    while (s[i] != _T('\0')) {
        switch (s[i]) {
        case _T('#'):
            if (!GridPresent) {
                GridPresent = true;
                GridPos     = i;
            }
            break;
        case _T(':'):
            ColonPresent = true;
            break;
        case _T('/'):
            SlashPresent = true;
            break;
        case _T('.'):
            PointPresent = true;
            break;
        case _T('@'):
            if (!DogPresent) {
                DogPresent = true;
                DogPos     = i;
            }
            break;
        }
        i++;
    }
    lstrcpy( szTempStr, s );
    if (GridPresent) {
        if (!(ColonPresent & SlashPresent & PointPresent & !DogPresent))
            return (false);
        if (GridPos > MAX_DOMAIN_SIZE)
            return (false);
        lstrcpyn( szDomain, s, GridPos+1 );
        lstrcpy( szTempStr, s + GridPos + 1);
    }
    if (DogPresent && !PointPresent) {
        lstrcpyn( szDomain, szTempStr + DogPos + 1, MAX_DOMAIN_SIZE );
        szTempStr[DogPos] = _T('\0');
    }
    if (!SlashPresent) {
        #ifdef SIGNED_ADDR
        wsprintf( szFullAddr, _T("%ld/%s"), g_vars.m_addrMain.Net, szTempStr );
        #else
        wsprintf( szFullAddr, _T("%lu/%s"), g_vars.m_addrMain.Net, szTempStr );
        #endif
        lstrcpy( szTempStr, szFullAddr );
    }
    if (!ColonPresent) {
        #ifdef SIGNED_ADDR
        wsprintf( szFullAddr, _T("%ld:%s"), g_vars.m_addrMain.Zone, szTempStr );
        #else
        wsprintf( szFullAddr, _T("%lu:%s"), g_vars.m_addrMain.Zone, szTempStr );
        #endif
    }
    else
        lstrcpy( szFullAddr, szTempStr );

    if (!ParseFTNAddress( szFullAddr, Addr ))
        return (false);

    if (GridPresent)
        lstrcpy( Addr.Domain, szDomain );
    else if (!DogPresent) {
	    // let's find link with the same zone
		PLink	link = NULL;
		CLinksManager::CIterator	it = CLinksManager::getInstance()->begin();
		while (it != CLinksManager::getInstance()->end()) {
			if ((*it)->getAddress().Zone == Addr.Zone) {
				link = (*it);
			    break;
		    }
			++it;
	    }

	    if (link != NULL)
		    lstrcpy( Addr.Domain, link->getAddress().Domain );
	    else
		    lstrcpy( Addr.Domain, g_vars.m_addrMain.Domain );
    }
    else if (!PointPresent) {
        lstrcpy( Addr.Domain, szDomain );
    }

    return (*(Addr.Domain) != _T('\0'));
}


bool CTic::Read(LPCTSTR szTicFileName) {
	int		i;
	SFTNAddress	SeenbyAddr;
	TCHAR		szLastError[256];

	// remember tic name
	_tcsncpy( fullName, szTicFileName, MAX_PATH - 1 );
	fullName[MAX_PATH] = TEXT('\0');
	name = fullName;
	for (i = 0; fullName[i] != TEXT('\0'); i++) {
		if (fullName[i] == PATH_SEPARATOR)
			name = fullName+i+1;
	}

	vector<tstring>	vsSeenBy;
	vector<int>	viSeenByLine;

	// variables for processing of CRC

	TCHAR		cNumber;


	Free();

	CCfgParser	CfgParser( szTicFileName );
	if (CfgParser.Failed()) {
		return false;
	}

    while (!CfgParser.Eof()) {
        CfgParser.ReadLine();
        if (!CfgParser.Failed() && CfgParser.Count() > 0) {
            switch (getTicLexemID( CfgParser[0] )) {
            /*************************
              Area (required, create)
             *************************/
            case TIC_LEXEM_AREA: {
                #ifdef TIC_DEBUG_INFO
                cout << "Parse Area\n";
                #endif
                if (*m_szArea != '\0') {
					sprintf( szLastError, "duplicate AREA tag in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (CfgParser.Count() > 2) {
					sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (CfgParser.Count() != 2) {
					sprintf( szLastError, "too few values in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (strlen(CfgParser[1]) > AREA_NAME_SIZE-1) {
					sprintf( szLastError, "too long area name in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                strcpy( m_szArea, CfgParser[1] );
				break;
            }
            /**********
              AreaDesc
             **********/
            case TIC_LEXEM_AREADESC: {
                #ifdef TIC_DEBUG_INFO
                cout << "Parse AreaDesc\n";
                #endif
                if (*m_szAreaDesc != '\0') {
					sprintf( szLastError, "duplicate AREADESC tag in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
				// if no data ignore it
                if (CfgParser.Count() > 1) {
					i = strlen( CfgParser.String() + 9 );
					if (i > AREA_DESC_SIZE-1) {
						// cout << "Warning: area description was shortened\n";
						i = AREA_DESC_SIZE-1;
					}
					memcpy( m_szAreaDesc, CfgParser.String() + 9, i );
					m_szAreaDesc[i] = '\0';
				}
                break;
            }
            /**************************
              File (required, created)
             **************************/
            case TIC_LEXEM_FILE: {
                #ifdef TIC_DEBUG_INFO
                cout << "Parse File\n";
                #endif
                if (*m_szFile != '\0') {
					sprintf( szLastError, "duplicate FILE tag in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (CfgParser.Count() > 2) {
					sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (CfgParser.Count() != 2) {
					sprintf( szLastError, "too few values in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                if (strlen(CfgParser[1]) > MAX_PATH-1) {
					sprintf( szLastError, "too long filename in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
                strcpy( m_szFile, CfgParser[1] );
                break;
            }
            /********************
              FullName (created)
             ********************/
            case TIC_LEXEM_FULLNAME: {
                #ifdef TIC_DEBUG_INFO
                cout << "Parse FullName\n";
                #endif
                if (*m_szFullName != '\0') {
				    sprintf( szLastError, "duplicate FULLNAME tag in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
				}
				if (CfgParser.Count() > 1) {
				    if (CfgParser.Count() > 2) {
						sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
						m_sLastError = szLastError;
						return false;
					}
					if (strlen(CfgParser[1]) > MAX_PATH-1) {
						sprintf( szLastError, "too long filename in line %d", CfgParser.GetLine() );
						m_sLastError = szLastError;
						return false;
					}
				    strcpy( m_szFullName, CfgParser[1] );
					if (m_szFullName[0] == '"')
						RemoveBoundaryQuotes( m_szFullName );
					if (m_szFullName[0] == '\0') {
						sprintf( szLastError, "filename absent in line %d", CfgParser.GetLine() );
						m_sLastError = szLastError;
						return false;
					}
				}
                break;
            }
            /**********
              LongName
             **********/
            case TIC_LEXEM_LONGNAME: {
                #ifdef TIC_DEBUG_INFO
                cout << "Parse LongName\n";
                #endif
                if (*m_szLongName != '\0') {
					sprintf( szLastError, "duplicate LONGNAME tag in line %d", CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
		    if (CfgParser.Count() > 1) {
			    if (CfgParser.Count() > 2) {
				    sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    if (strlen(CfgParser[1]) > MAX_PATH-1) {
				    sprintf( szLastError, "too long filename in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    strcpy( m_szLongName, CfgParser[1] );
			    if (m_szLongName[0] == '"')
				RemoveBoundaryQuotes( m_szLongName );
			    if (m_szLongName[0] == '\0') {
				    sprintf( szLastError, "filename absent in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
		    }
                    break;
                }
                /*************************
                  CRC (required, created)
                 *************************/
                case TIC_LEXEM_CRC: {

			#ifdef TIC_DEBUG_INFO
			cout << "Parse CRC\n";
			#endif

			// check number of arguments
			if (CfgParser.Count() > 2) {
				_stprintf( szLastError, TEXT("extra values in line %d"), CfgParser.GetLine() );
				m_sLastError = szLastError;
				return false;
			}
			if (CfgParser.Count() < 2) {
				_stprintf( szLastError, TEXT("no values in line %d"), CfgParser.GetLine() );
				m_sLastError = szLastError;
				return false;
			}

			// avoid duplicates
			if (m_bCRC32Present) {
				_stprintf( szLastError, TEXT("duplicate CRC tag in line %d"), CfgParser.GetLine() );
				m_sLastError = szLastError;
				return false;
			}

			// parse CRC32
			
			for (m_uiCRC32 = 0, i = 0; i < strlen(CfgParser[1]); i++) {
				cNumber = ToUpper(CfgParser[1][i]);
				if (!isalnum(cNumber) || cNumber > TEXT('F')) {
					_stprintf( szLastError, TEXT("Incorrect CRC value in line %d"), CfgParser.GetLine() );
					m_sLastError = szLastError;
					return false;
				}
				m_uiCRC32 <<= 4;
				if (isalpha(cNumber))
					m_uiCRC32 += cNumber - TEXT('A') + 10;
				else
					m_uiCRC32 += cNumber - TEXT('0');
			}

			m_bCRC32Present = true;

			break;
                }
                /**************************************
                  Magic (should not be passed through)
                 **************************************/
                case TIC_LEXEM_MAGIC: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Magic\n";
                    #endif
                    if (*m_szMagic != '\0') {
			    sprintf( szLastError, "duplicate MAGIC tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
		    if (CfgParser.Count() > 1) {
			    if (CfgParser.Count() > 2) {
				    sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    if (strlen(CfgParser[1]) > MAX_PATH-1) {
				    sprintf( szLastError, "too long filename in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    strcpy( m_szMagic, CfgParser[1] );
		    }
                    break;
                }
                /**********
                  Replaces
                 **********/
                case TIC_LEXEM_REPLACES: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Replaces\n";
                    #endif
                    if (*m_szReplaces != '\0') {
			    sprintf( szLastError, "duplicate REPLACES tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
		    if (CfgParser.Count() > 1) {
			    if (CfgParser.Count() > 2) {
				    sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    if (strlen(CfgParser[1]) > MAX_PATH-1) {
				    sprintf( szLastError, "too long filename in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    strcpy( m_szReplaces, CfgParser[1] );
		    }
                    break;
                }
                /****************
                  Desc (created)
                 ****************/
                case TIC_LEXEM_DESC: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Desc\n";
                    #endif
                    if (CfgParser.String()[0] == ' ') { 
			    sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
		    if (CfgParser.Count() > 1) {
			    if (strlen(CfgParser.String() + 5) > 256) {
				    sprintf( szLastError, "too long description in line %d", CfgParser.GetLine() );
				    m_sLastError = szLastError;
				    return false;
			    }
			    if (*m_szDesc == _T('\0')) {
				lstrcpy( m_szDesc, CfgParser.String() + 5 );
			    }
			    else {
				m_vsDescLines.push_back( CfgParser.String() + 5 );
			    }
		    }
                    break;
                }
                /*****************
                  LDesc (created)
                 *****************/
                case TIC_LEXEM_LDESC: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse LDesc\n";
                    #endif
                    if (CfgParser.String()[0] == ' ') {
			    sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
		    if (CfgParser.Count() > 1) {
			m_listLDesc.push_back(CfgParser.String() + 6);
		    }
                    break;
                }
                /************************************
                  Size (created, should be required)
                 ************************************/
                case TIC_LEXEM_SIZE: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Size\n";
                    #endif
                    if (m_iSize != -1) {
			    sprintf( szLastError, "duplicate SIZE tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() > 2) {
			    sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
		    if (CfgParser.Count() < 2) {
			    sprintf( szLastError, "no values in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (sscanf( CfgParser[1], "%ld", &m_iSize ) != 1) {
			    sprintf( szLastError, "error parsering size value in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    break;
                }
                /****************************
                  Origin (required, created)
                 ****************************/
                case TIC_LEXEM_ORIGIN: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Origin\n";
                    #endif
                    if (m_addrOrigin.Zone != 0) {
			    sprintf( szLastError, "duplicate ORIGIN tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() != 2) {
			    sprintf( szLastError, "incorrect number of values (not 1) in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (!ParseAddr( CfgParser[1], m_addrOrigin )) {
			    sprintf( szLastError, "failed to parse ORIGIN address in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    break;
                }
                /**************************
                  From (required, created)
                 **************************/
                case TIC_LEXEM_FROM: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse From\n";
                    #endif
                    if (m_addrFrom.Zone != 0) {
			    sprintf( szLastError, "duplicate FROM tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() > 3) {
			    sprintf( szLastError, "extra values in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() < 2) {
			    sprintf( szLastError, "no values in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (!ParseAddr( CfgParser[1], m_addrFrom )) {
			    sprintf( szLastError, "failed to parse FROM address in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() == 3)
                        strcpy( m_szFromPwd, CfgParser[2] );
                    break;
                }
                /**********************************
                  To (should not be passed through
                 **********************************/
                case TIC_LEXEM_TO: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse To\n";
                    #endif
                    if (m_addrTo.Zone != 0) {
			    sprintf( szLastError, "duplicate TO tag in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (CfgParser.Count() != 2) {
			    sprintf( szLastError, "incorrect number of values (not 1) in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (!ParseAddr( CfgParser[1], m_addrTo )) {
			    sprintf( szLastError, "failed to parse TO address in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    break;
                }
                /***********************************************************
                  Created (required, created, should not be passed through)
                 ***********************************************************/
                case TIC_LEXEM_CREATED: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Created\n";
                    #endif
                    if (*m_szCreatedBy != '\0')
                        break;
                    if (CfgParser.String()[0] == ' ') {
			    sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );                    
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (strlen(CfgParser.String() + 8) > 256) {
			    sprintf( szLastError, "too long CREATED value in line %d", CfgParser.GetLine() );                    
			    m_sLastError = szLastError;
			    return false;
		    }
                    if (strnicmp( CfgParser.String(), "Created by ", 11 ) == 0)
                        strcpy( m_szCreatedBy, CfgParser.String() + 11 );
                    else
                        strcpy( m_szCreatedBy, CfgParser.String() + 8 );
                    break;
                }
                /*****
                  Via
                 *****/
                case TIC_LEXEM_VIA: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Via\n";
                    #endif
		    if (CfgParser.Count() > 1) {
			    if (CfgParser.String()[0] == ' ') {
				    sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );                    
				    m_sLastError = szLastError;
				    return false;
			    }
			    m_listVia.push_back(CfgParser.String() + 4);
		    }
                    break;
                }
                /**************************
                  Path (created, required)
                 **************************/
                case TIC_LEXEM_PATH: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Path\n";
                    #endif
		    if (CfgParser.Count() > 1) {
			    if (CfgParser.String()[0] == ' ') {
				    sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );                    
				    m_sLastError = szLastError;
				    return false;
			    }
			    m_listPath.push_back( CfgParser.String() + 5 );
		    }
                    break;
                }
                /****************************
                  SeenBy (required, created)
                 ****************************/
                case TIC_LEXEM_SEENBY: {
			#ifdef TIC_DEBUG_INFO
			cout << "Parse SeenBy\n";
			#endif

			// check for nonempty SeenBy
			if (CfgParser.Count() > 1) {
				// keep SeenBy for future analysis
				vsSeenBy.push_back( CfgParser.String() + CfgParser.Where(1) );
				viSeenByLine.push_back( CfgParser.GetLine() );
			}

			/*
			if (CfgParser.String()[0] == ' ') {
				sprintf( szLastError, "extra spaces in line %d", CfgParser.GetLine() );
				m_sLastError = szLastError;
				return (0);
			}
			SFTNAddress	addr;
			bool	fParsed = ParseAddr( CfgParser.String() + 7, addr );
			if (!fParsed) {
				sprintf( szLastError, "failed to parse SEENBY address in line %d", CfgParser.GetLine() );
				m_sLastError = szLastError;
				return (0);
			}		    
			m_listSeenby.push_back( addr );
			*/
			break;
                }
                /***************
                  Pw (required)
                 ***************/
                case TIC_LEXEM_PW: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse PW\n";
                    #endif
                    if (*m_szPW != '\0')
                        break;
                    if (CfgParser.Count() != 2) {
			    sprintf( szLastError, "wrong number of values (not 1) in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    strcpy( m_szPW, CfgParser[1] );
                    break;
                }
                /*****
                  Pwd
                 *****/
                case TIC_LEXEM_PWD: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse PWD\n";
                    #endif
                    if (*m_szPW != '\0')
                        break;
                    if (CfgParser.Count() != 2) {
			    sprintf( szLastError, "wrong number of values (not 1) in line %d", CfgParser.GetLine() );
			    m_sLastError = szLastError;
			    return false;
		    }
                    strcpy( m_szPW, CfgParser[1] );
                    break;
                }
                /****************
                  ReceiptRequest
                 ****************/
                case TIC_LEXEM_RECEIPTREQUEST: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse ReceiptRequest\n";
                    #endif
                    break;
                }
                /*************
                  Destination
                 *************/
                case TIC_LEXEM_DESTINATION: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse Destination\n";
                    #endif
                    break;
                }
                /****************
                  Unknown string
                 ****************/
                default: {
                    #ifdef TIC_DEBUG_INFO
                    cout << "Parse unknown\n";
                    #endif
                    m_listUnknown.push_back( CfgParser.String() );
                }
            }
        }
    }
    if (!CfgParser.Eof())
        return false;
    // check tic 
    if (*m_szArea == _T('\0')) {
	    m_sLastError = _T("Area tag absent");
	    return false;
    }
    if (*m_szFile == _T('\0')) {
	    m_sLastError = _T("File tag absent");
	    return false;
    }
    if (m_addrOrigin.Zone == 0) {
	    m_sLastError = _T("Origin tag absent");
	    return false;
    }
    if (m_addrFrom.Zone == 0) {
	    m_sLastError = _T("From tag absent");
	    return false;
    }
    if (*m_szPW == _T('\0')) {
	    m_sLastError = _T("PW tag absent");
	    return false;
    }

	/*
	
	  We leave crc32 processing for our callers

	if (!CRCDefined) {
		if (g_warnings.m_flagIncorrectCRC32) {
			g_vars.m_log << TEXT("[!] ") << szTicFileName << TEXT(": CRC undefined\n");
		}
		else  {
			m_sLastError = TEXT("CRC tag absent");
			return false;
		}
	}
	*/
	
	// if there is no FullName but there is LongName, copy it to FullName
	if (*m_szFullName == _T('\0') && *m_szLongName != _T('\0'))
		lstrcpy( m_szFullName, m_szLongName );

	// if there are more than 1 Desc line, we can concatenate them with LDesc lines
	if (g_vars.m_flagConcatDescriptions && !m_vsDescLines.empty()) {
		for (int iLine = 0; iLine < m_vsDescLines.size(); iLine++) {
			m_listLDesc.insert( m_listLDesc.begin(), m_vsDescLines[iLine] );
		}
	}

	// process seenby list

	if (!ParseSeenByStrings( vsSeenBy, viSeenByLine ))
		return false;

	m_sLastError = _T("");

	return true;
}


bool CTic::ParseSeenByStrings(vector<tstring>& vsSeenBy, vector<int> viSeenByLine) {
	int				nCount = vsSeenBy.size();
	int				i;
	list<SFTNAddress>::iterator	it;
	SFTNAddress			addr;
	SAddr				prevAddr;
	bool				bSuccess = true;

	for (i = 0; i < nCount && bSuccess; i++) {
		// get seenby line
		LPCTSTR	pszSeenBy = vsSeenBy[i].c_str();
		// clear prevous address
		prevAddr.m_iZone  = -1;
		prevAddr.m_iNet   = -1;
		prevAddr.m_iNode  = -1;
		prevAddr.m_iPoint = -1;
		prevAddr.m_sDomain.clear();
		while (*pszSeenBy != TEXT('\0')) {
			// skip spaces and tabs
			while (*pszSeenBy == TEXT(' ') || *pszSeenBy == TEXT('\011'))
				pszSeenBy++;
			if (*pszSeenBy == TEXT('\0'))
				break;
			// keep start of address
			LPCTSTR	pszAddr = pszSeenBy;
			// go throught a current address
			TCHAR	cCurrent;
			do {
				pszSeenBy++;
				cCurrent = *pszSeenBy;
			} while ((cCurrent != TEXT('\0')) && (cCurrent != TEXT(' ')) && (cCurrent != TEXT('\011')));
			// try to parse address
			if (!ParseSeenByAddr( pszAddr, pszSeenBy - pszAddr, addr, prevAddr )) {
				TCHAR	szLastError[128];
				_stprintf( szLastError, TEXT("failed to parse SEENBY address in line %d"), viSeenByLine[i] );
				m_sLastError = szLastError;

				// DEBUG
				// cerr << szLastError << endl;

				bSuccess = false;
				break;
			}
			// check for duplicate address

			for (it = m_listSeenby.begin(); it != m_listSeenby.end(); it++) {
				if (AddrCmp( (*it), addr ) == 0)
					break;
			}

			if (it == m_listSeenby.end()) {
				// keep address
				prevAddr.m_iZone   = addr.Zone;
				prevAddr.m_iNet    = addr.Net;
				prevAddr.m_iNode   = addr.Node;
				prevAddr.m_iPoint  = addr.Point;
				prevAddr.m_sDomain = addr.Domain;

				// address is unique
				m_listSeenby.push_back( addr );

				// DEBUG

				/*
				TCHAR szAddr[64];
				addr.ToString( szAddr );
				cout << i << ": " << szAddr << endl;
				*/
			}

		}
	}
	return (bSuccess);
}


// states for seenby parser

enum ParseStates {
	PS_1ST,
	PS_ZONE,
	PS_NET,
	PS_NODE,
	PS_POINT,
	PS_DOMAIN,
	PS_ERROR,
	PS_EXIT
};

bool CTic::ParseSeenByAddr(LPCTSTR psAddr, int iAddrStrLen, SFTNAddress& addr, SAddr& prevAddr) {
	int	nState = PS_1ST;
	LPCTSTR	psAccumulator = psAddr;
	tstring	sDomain;
	tstring	sZone;
	tstring	sNet;
	tstring	sNode;
	tstring	sPoint;

	while ((nState != PS_ERROR) && (nState != PS_EXIT)) {
		TCHAR	cCurrent = *psAddr;
		switch (nState) {

		case PS_1ST:

			switch (cCurrent) {
			case TEXT('#'):
				// keep domain
				sDomain.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_ZONE;
				psAccumulator = psAddr;
				break;
			case TEXT('@'):
				nState = PS_ERROR;
				break;
			case TEXT(':'):
				// keep zone
				sZone.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_NET;
				psAccumulator = psAddr;
				break;
			case TEXT('/'):
				// keep net
				sNet.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_NODE;
				psAccumulator = psAddr;
				break;
			case TEXT('.'):
				// keep node
				sNode.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_POINT;
				psAccumulator = psAddr;
				break;
			default:
				if (iAddrStrLen == 0) {
					// keep node
					sNode.assign( psAccumulator, psAddr - psAccumulator );
					// change state
					nState        = PS_EXIT;
				}
				else
				if (isalnum(cCurrent)) {
					// accumulate
					psAddr++;
					iAddrStrLen--;
				}
				else {
					nState = PS_EXIT;
				}
				break;
			}
			break;

		case PS_ZONE:

			switch (cCurrent) {
			case TEXT(':'):
				// keep zone
				sZone.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_NET;
				psAccumulator = psAddr;
				break;
			default:
				if (iAddrStrLen == 0) {
					nState = PS_ERROR;
				}
				else
				if (isdigit(cCurrent)) {
					// accumulate
					psAddr++;
					iAddrStrLen--;
				}
				else {
					nState = PS_ERROR;
				}
				break;
			}
			break;

		case PS_NET:

			switch (cCurrent) {
			case TEXT('/'):
				// keep net
				sNet.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_NODE;
				psAccumulator = psAddr;
				break;
			default:
				if (iAddrStrLen == 0) {
					nState = PS_ERROR;
				}
				else
				if (isdigit(cCurrent)) {
					// accumulate
					psAddr++;
					iAddrStrLen--;
				}
				else {
					nState = PS_ERROR;
				}
				break;
			}
			break;

		case PS_NODE:

			switch (cCurrent) {
			case TEXT('.'):
				// keep node
				sNode.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_POINT;
				psAccumulator = psAddr;
				break;
			case TEXT('@'):
				// keep node
				sNode.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_DOMAIN;
				psAccumulator = psAddr;
				break;
			default:
				if (iAddrStrLen == 0) {
					// keep node
					sNode.assign( psAccumulator, psAddr - psAccumulator );
					// change state
					nState        = PS_EXIT;
				}
				else
				if (isdigit(cCurrent)) {
					// accumulate
					psAddr++;
					iAddrStrLen--;
				}
				else {
					nState = PS_ERROR;
				}
				break;
			}
			break;

		case PS_POINT:

			switch (cCurrent) {
			case TEXT('@'):
				// keep point
				sPoint.assign( psAccumulator, psAddr - psAccumulator );
				psAddr++;
				iAddrStrLen--;
				// change state
				nState        = PS_DOMAIN;
				psAccumulator = psAddr;
				break;
			default:
				if (iAddrStrLen == 0) {
					// keep point
					sPoint.assign( psAccumulator, psAddr - psAccumulator );
					// change state
					nState        = PS_EXIT;
				}
				else
				if (isdigit(cCurrent)) {
					// accumulate
					psAddr++;
					iAddrStrLen--;
				}
				else {
					nState = PS_ERROR;
				}
				break;
			}
			break;

		case PS_DOMAIN:

			if (iAddrStrLen == 0) {
				// keep domain
				sDomain.assign( psAccumulator, psAddr - psAccumulator );
				// change state
				nState        = PS_EXIT;
			}
			else
			if (cCurrent == TEXT('.') || isalnum(cCurrent)) {
				// accumulate
				psAddr++;
				iAddrStrLen--;
			}
			else {
				nState = PS_ERROR;
			}
			break;
		}
	}

	if (nState == PS_ERROR)
		return (false);

	// now we have some parts of address
	// let's parse them

	int	iZone  = -1;
	int	iNet   = -1;
	int	iNode  = -1;
	int	iPoint = -1;

	// parse zone
	if (!sZone.empty()) {
		if (!StrToInt( sZone.c_str(), iZone ))
			return (false);
		if (iZone < 0 || iZone > 65535)
			return (false);
	}
	// parse net
	if (!sNet.empty()) {
		if (!StrToInt( sNet.c_str(), iNet ))
			return (false);
		if (iNet < 0 || iNet > 65535)
			return (false);
	}
	// parse node
	if (!sNode.empty()) {
		if (!StrToInt( sNode.c_str(), iNode ))
			return (false);
		if (iNode < 0 || iNode > 65535)
			return (false);
	}
	// parse point
	if (!sPoint.empty()) {
		if (!StrToInt( sPoint.c_str(), iPoint ))
			return (false);
		if (iPoint < 0 || iPoint > 65535)
			return (false);
	}
	
	if (!sDomain.empty()) {
		// we should have FQFA

		if (iZone == -1 || iNet == -1 || iNode == -1)
			// some component of address is absent
			return (false);

		addr.Zone  = iZone;
		addr.Net   = iNet;
		addr.Node  = iNode;
		addr.Point = (iPoint == -1 ? 0 : iPoint);
		lstrcpyn( addr.Domain, sDomain.c_str(), MAX_DOMAIN_SIZE );

		return (true);
	}

	// we have no domain

	if (iZone != -1) {
		// we have a zone

		if (iNet == -1 || iNode == -1) {
			return (false);
		}

		addr.Zone  = iZone;
		addr.Net   = iNet;
		addr.Node  = iNode;
		addr.Point = (iPoint == -1 ? 0 : iPoint);
		if (prevAddr.m_sDomain.empty()) {
			// get domain from "From" field
			lstrcpyn( addr.Domain, m_addrFrom.Domain, MAX_DOMAIN_SIZE );
		}
		else {
			lstrcpyn( addr.Domain, prevAddr.m_sDomain.c_str(), MAX_DOMAIN_SIZE );
		}
		return (true);
	}

	// we have no zone and domain

	if (iNode == -1)
		return (false);

	addr.Node = iNode;
	addr.Point = (iPoint == -1 ? 0 : iPoint);
	if (prevAddr.m_iZone == -1) {
		// get zone and domain from "From"
		addr.Zone = m_addrFrom.Zone;
		lstrcpyn( addr.Domain, m_addrFrom.Domain, MAX_DOMAIN_SIZE );
		// set net
		if (iNet == -1)
			addr.Net = m_addrFrom.Net;
		else
			addr.Net = iNet;
	}
	else {
		// get zone and domain from prevous address
		addr.Zone = prevAddr.m_iZone;
		lstrcpyn( addr.Domain, prevAddr.m_sDomain.c_str(), MAX_DOMAIN_SIZE );
		// set net
		if (iNet == -1)
			addr.Net = prevAddr.m_iNet;
		else
			addr.Net = iNet;
	}

	return (true);
}


void CTic::PrintInfo() {
    cout << "Tic file information\n";
    cout << "Area = " << m_szArea << endl;
    cout << "File = " << m_szFile << endl;
    cout << "Desc = " << m_szDesc << endl;
    cout << "Origin = " << m_addrOrigin.Zone << ":" << m_addrOrigin.Net << "/"
         << m_addrOrigin.Node << "." << m_addrOrigin.Point << "@" << m_addrOrigin.Domain << endl;
    cout << "From = " << m_addrFrom.Zone << ":" << m_addrFrom.Net << "/" << m_addrFrom.Node
         << "." << m_addrFrom.Point << "@" << m_addrFrom.Domain << endl;
    cout << "To = " << m_addrTo.Zone << ":" << m_addrTo.Net << "/" << m_addrTo.Node
         << "." << m_addrTo.Point << "@" << m_addrTo.Domain << endl;
    cout << "PW = " << m_szPW << endl;
}


bool CTic::Forward(PLink pLink, PArea area, LPCTSTR szFilePath, vector<PLink>& forwards) {
	const int	COPY_TRY_LIMIT = 16;	// we'll make not more than COPY_TRY_LIMIT tries to copy file
										// into filebox
	// prepare outbound type and flavour type for forward
	OutboundType	outboundType;
	FlavourType		flavourType;

	flavourType  = pLink->getFlavourType();
	if (area->isOutboundDefined()) {
		outboundType = area->getOutboundType();
		if (outboundType == OT_FILE_BOXES) {
			if (flavourType != FT_HOLD) {
				flavourType = FT_NORMAL;
			}
		}
	} else {
		outboundType = pLink->getOutboundType();
	}

	time_t  timestamp;
	TCHAR   szTicFileName[MAX_PATH];
	TCHAR   szFileName[MAX_PATH];
	TCHAR   szShortFileName[MAX_PATH];

	HANDLE	hFile;

	TCHAR	szBuf[STR_BUF_SIZE];

	int     i, j;

	tstring	fileBoxPath;	// for fileBoxes outbound type

	// form full file name
	lstrcpy( szFileName, szFilePath );
	if (*m_szFullName != _T('\0')) {
		lstrcat( szFileName, m_szFullName );
	} else if (*m_szLongName != _T('\0')) {
		lstrcat( szFileName, m_szLongName );
	} else {
		lstrcat( szFileName, m_szFile );
	}

	// form tic name
	switch (outboundType) {
		case OT_BINKLEY:
			wsprintf( szTicFileName, _T("%sad%06lu.tic"), g_vars.m_szNewTicsPath, m_TicNamer.getFreeNumber() );
			break;
		case OT_FILE_BOXES:
			fileBoxPath = pLink->getFileBoxPath();
			wsprintf( szTicFileName, _T("%s\\ad%06lu.tic"), fileBoxPath.c_str(), m_TicNamer.getFreeNumber() );
			break;
	}

	if (outboundType == OT_FILE_BOXES) {
		// place file into filebox
		tstring	destFileName = fileBoxPath + TEXT("\\");
		if (*m_szFullName != TEXT('\0')) {
			destFileName += m_szFullName;
		} else if (*m_szLongName != TEXT('\0')) {
			destFileName += m_szLongName;
		} else {
			destFileName += m_szFile;
		}

		// check destination file existance
		if (_taccess( destFileName.c_str(), 0 ) == 0 || errno != ENOENT) {
			// log warning
			g_vars.m_log << CLog::ResetSingle(new CLog::StdErr())
							<< TEXT("[warning] destination file already exists, will be overwritten:\n")
							<< TEXT("[warning] ") << destFileName << TEXT("\n")
							<< CLog::Pop();
		}
		int				copyTryCount = 0;
		FileOpResult	foResult     = FO_FAILED;
		do {
			// ensure that path exists
			_tmkdir( fileBoxPath.c_str() );
			// try to copy file
			++copyTryCount;
			foResult = MyCopyFile( szFileName, destFileName.c_str() );
		} while (foResult == FO_DEST_UNAVAILABLE && copyTryCount < COPY_TRY_LIMIT);
		if (foResult != FO_SUCCEEDED) {
			// log copy error
			g_vars.m_log << CLog::ResetSingle(new CLog::StdOut())
						 << TEXT("[error] failed to copy file into filebox\n")
						 << TEXT("[error]        source file: ") << szFileName << TEXT("\n")
						 << TEXT("[error]   destination file: ") << destFileName.c_str() << TEXT("\n")
						 << CLog::Pop();
			return false;
		}
		// adjust full file name
		lstrcpy( szFileName, destFileName.c_str() );
	}

	// create tic

	// open file for writing
	hFile = CreateFile( szTicFileName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, 
			FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
		return (false);

	// Area

	_stprintf( szBuf, _T("Area %s\r\n"), m_szArea );
	WriteCharString( hFile, szBuf );

	// AreaDesc

	if (*m_szAreaDesc != _T('\0')) {
		_stprintf( szBuf, _T("AreaDesc %s\r\n"), m_szAreaDesc );
		WriteCharString( hFile, szBuf );
	}

	// File

	ConvertANSINameToShort( szFileName, szShortFileName );
	// get only file name without path
	for (i = 0, j = -1; szShortFileName[i] != _T('\0'); i++) {
		if (szShortFileName[i] == _T('\\'))
			j = i;
	}
	j++;
	_stprintf( szBuf, _T("File %s\r\n"), szShortFileName + j );
	WriteCharString( hFile, szBuf );

	// FullName

	// check full file name
	if (*m_szFullName != _T('\0'))
		lstrcpy( szFileName, m_szFullName );
	else if (*m_szLongName != _T('\0'))
		lstrcpy( szFileName, m_szLongName );
	else
		lstrcpy( szFileName, m_szFile );
	if (lstrcmp( szFileName, szShortFileName + j) != 0) {
		// we need to use FullName tag
		// we also write LongName tag
		// check spaces
		for (i = 0; szFileName[i] != _T('\0'); i++) {
			if (szFileName[i] == _T(' '))
				break;
		}
		if (szFileName[i] == _T(' ')) { // contains space(s)
			_stprintf( szBuf, _T("FullName \"%s\"\r\n"), szFileName );
		}
		else
			_stprintf( szBuf, _T("FullName %s\r\n"), szFileName );
		WriteCharString( hFile, szBuf );
		// overwrite 'Full' with 'Long'
		::CopyMemory( szBuf, _T("Long"), 4*sizeof(TCHAR) );
		// write LongName tag
		WriteCharString( hFile, szBuf );
	}

	// Size

	if (m_iSize != -1) {
		_stprintf( szBuf, _T("Size %d\r\n"), m_iSize );
		WriteCharString( hFile, szBuf );
	}

	// CRC

	_stprintf( szBuf, _T("CRC %08lX\r\n"), m_uiCRC32 );
	WriteCharString( hFile, szBuf );

	// Origin

	_stprintf( szBuf, _T("Origin %hu:%hu/%hu"), m_addrOrigin.Zone, m_addrOrigin.Net,
		m_addrOrigin.Node );
	if (m_addrOrigin.Point != 0)
		_stprintf( szBuf + lstrlen(szBuf), _T(".%hu"), m_addrOrigin.Point );
	if (lstrcmpi( m_addrOrigin.Domain, pLink->getAddress().Domain) != 0 ||
		lstrcmpi( m_addrOrigin.Domain, g_vars.m_addrMain.Domain) != 0)
		_stprintf( szBuf + lstrlen(szBuf), _T("@%s"), m_addrOrigin.Domain );
	lstrcat( szBuf + lstrlen(szBuf), _T("\r\n") );
	WriteCharString( hFile, szBuf );

	// From

	const SFTNAddress&	ourAddr = pLink->getOurAddress();

	_stprintf( szBuf, _T("From %hu:%hu/%hu"), ourAddr.Zone,
		ourAddr.Net, ourAddr.Node );
	if (ourAddr.Point != 0)
		_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"), ourAddr.Point );
	if (lstrcmpi( ourAddr.Domain, pLink->getAddress().Domain) != 0)
		_stprintf( szBuf+lstrlen(szBuf), _T("@%s"), ourAddr.Domain );
	lstrcat( szBuf, _T("\r\n") );
	WriteCharString( hFile, szBuf );

	// To

	_stprintf( szBuf, _T("To %hu:%hu/%hu"), pLink->getAddress().Zone, pLink->getAddress().Net,
		pLink->getAddress().Node );
	if (pLink->getAddress().Point != 0)
		_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"), pLink->getAddress().Point );
	if (lstrcmpi( ourAddr.Domain, pLink->getAddress().Domain) != 0)
		_stprintf( szBuf+lstrlen(szBuf), _T("@%s"), pLink->getAddress().Domain );
	lstrcat( szBuf, _T("\r\n") );
	WriteCharString( hFile, szBuf );

	// Desc

	if (*m_szDesc != _T('\0')) {
		_stprintf( szBuf, _T("Desc %s\r\n"), m_szDesc );
		WriteCharString( hFile, szBuf );
	}

	// LDesc

	{
		vector<tstring>::iterator	i;
		for (i = m_listLDesc.begin(); i != m_listLDesc.end(); i++) {
			_stprintf( szBuf, _T("LDesc %s\r\n"), (*i).c_str() );
			WriteCharString( hFile, szBuf );
		}
	}

	// Created by

	_stprintf( szBuf, _T("Created by %s Copyright (c) Andrey Kuprishov (2:5023/27@Fidonet)\r\n"),
		CAdrenalin::getName().c_str() );
	WriteCharString( hFile, szBuf );

	// Replaces

	if (*m_szReplaces != _T('\0')) {
		_stprintf( szBuf, _T("Replaces %s\r\n"), m_szReplaces );
		WriteCharString( hFile, szBuf );
	}

	// Seenby

	{
		list<SFTNAddress>::iterator	i;
		for (i = m_listSeenby.begin(); i != m_listSeenby.end(); i++) {
			_stprintf( szBuf, _T("Seenby %hu:%hu/%hu"), 
				(*i).Zone, (*i).Net, (*i).Node );
			if ((*i).Point != 0) {
				_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"), 
					(*i).Point );
			}
			if (lstrcmpi( (*i).Domain, pLink->getAddress().Domain) != 0 ||
				lstrcmpi( (*i).Domain, ourAddr.Domain) != 0)
			{
				_stprintf( szBuf+lstrlen(szBuf), _T("@%s"), (*i).Domain );
			}
			lstrcat( szBuf, _T("\r\n") );
			WriteCharString( hFile, szBuf );
		}
	}

	// output all forwards as Seenbys

	vector<PLink>::iterator	itLinked = forwards.begin();
	while (itLinked != forwards.end()) {
		PLink	link = (*itLinked);
		_stprintf( szBuf, _T("Seenby %hu:%hu/%hu"), 
			link->getAddress().Zone, 
			link->getAddress().Net,
			link->getAddress().Node );
		if (link->getAddress().Point > 0) {
			_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"), link->getAddress().Point );
		}
		if (lstrcmpi( link->getAddress().Domain, pLink->getAddress().Domain) != 0 ||
			lstrcmpi( link->getAddress().Domain, ourAddr.Domain) != 0)
		{
			_stprintf( szBuf+lstrlen(szBuf), _T("@%s"), link->getAddress().Domain );
		}
		lstrcat( szBuf, TEXT("\r\n") );
		WriteCharString( hFile, szBuf );
		++itLinked;
	}

	// output AKA for addressie if absent in seenbys

	if (!g_vars.m_fAllAKASeen) {
		list<SFTNAddress>::iterator	i;
		for (i = m_listSeenby.begin(); i != m_listSeenby.end(); i++) {
			if (AddrCmp( (*i), ourAddr ) == 0)
				break;
		}
		if (i == m_listSeenby.end()) {
			_stprintf( szBuf, _T("Seenby %hu:%hu/%hu"), 
				ourAddr.Zone, ourAddr.Net,
				ourAddr.Node );
			if (ourAddr.Point != 0)
				_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"),
					ourAddr.Point );
			if (lstrcmpi( ourAddr.Domain, pLink->getAddress().Domain) != 0)
				_stprintf( szBuf+lstrlen(szBuf), _T("@%s"), ourAddr.Domain );
			lstrcat( szBuf, _T("\r\n") );
			WriteCharString( hFile, szBuf );
		}    
	}

	// output Via lines

	{
		vector<tstring>::iterator	i;
		for (i = m_listVia.begin(); i != m_listVia.end(); i++) {
			_stprintf( szBuf, _T("Via %s\r\n"), (*i).c_str() );
			WriteCharString( hFile, szBuf );
		}
		if (*m_szCreatedBy != _T('\0')) {
			_stprintf( szBuf, _T("Via %hu:%hu/%hu"),
				m_addrFrom.Zone, m_addrFrom.Net, m_addrFrom.Node );
			if (m_addrFrom.Point != 0) {
				_stprintf( szBuf+lstrlen(szBuf), _T(".%hu"),
					m_addrFrom.Point );
			}
			_stprintf( szBuf+lstrlen(szBuf), _T("@%s %s\r\n"), 
				m_addrFrom.Domain, m_szCreatedBy );
			WriteCharString( hFile, szBuf );
		}
	}

	// output Path lines

	{
		vector<tstring>::iterator	i;
		for (i = m_listPath.begin(); i != m_listPath.end(); i++) {
			_stprintf( szBuf, _T("Path %s\r\n"), (*i).c_str() );
			WriteCharString( hFile, szBuf );
		}
	}

	TCHAR	szTime[32];
	timestamp = time(NULL);
	lstrcpy( szTime, _tasctime( gmtime( &timestamp )) );
	szTime[lstrlen(szTime)-1] = _T('\0');
	lstrcpy( szBuf, _T("Path ") );
	ourAddr.ToString( szBuf + lstrlen(szBuf) );
	_stprintf( szBuf+lstrlen(szBuf), _T(" %lu %s\r\n"), timestamp, szTime );
	WriteCharString( hFile, szBuf );

	// PW

	_stprintf( szBuf, _T("PW %s\r\n"), pLink->getPassword().c_str() );
	WriteCharString( hFile, szBuf );

	// output unknown lines

	{
		vector<tstring>::iterator	i;
		for (i = m_listUnknown.begin(); i != m_listUnknown.end(); i++) {
			_stprintf( szBuf, _T("%s\r\n"), (*i).c_str() );
			WriteCharString( hFile, szBuf );
		}
	}

	CloseHandle( hFile );
    
	if (outboundType == OT_BINKLEY) {
		// do BSO

		bool	busyFlagSet = true;
		m_BSOManager.setAddress( pLink->getAddress(), pLink->getFlavourType() );
		if (!m_BSOManager.setBusyFlag()) {
			m_BSOManager.setTempOutboundAddress( pLink->getAddress(), pLink->getFlavourType() );
			busyFlagSet = false;
		}

		hFile = CreateFile( m_BSOManager.getName(), GENERIC_WRITE, 0, NULL, OPEN_ALWAYS,
							FILE_ATTRIBUTE_NORMAL, NULL);
		if (hFile == INVALID_HANDLE_VALUE) {
			if (busyFlagSet) {
				m_BSOManager.clearBusyFlag();
			}
			_tunlink( szTicFileName );
			// TODO: log error
			return (false);
		}
		SetFilePointer( hFile, 0, NULL, FILE_END );

		wsprintf( szBuf, TEXT("%s\r\n"), szShortFileName );
		WriteCharString( hFile, szBuf );

		ConvertNameToShort( szTicFileName, szShortFileName );
		wsprintf( szBuf, TEXT("^%s\r\n"), szShortFileName );
		WriteCharString( hFile, szBuf );

		CloseHandle( hFile );

		if (busyFlagSet) {
			m_BSOManager.clearBusyFlag();
		}
	}

	return (true);
}
